//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.5
import QtQuick.Dialogs  1.2
import QtQuick.Layouts  1.2

import wesual.Ui             1.0
import wesual.Create        1.0
import wesual.Stage         1.0
import wesual.module.ProductManager 1.0
import wesual.module.MediaManager   1.0
import wesual.Controls       1.0
import wesual.Editor         1.0

import wesual.data.Core        1.0
import wesual.data.FoodService 1.0

Item {
    id : editor

    readonly property string label : qsTrId("5dfc83ce438ec875")

    property ListItemContainer container
    property Product           product  : container.item.data.product

    y : 10

    QtObject {
        id : p_

        property Item shareButton : null
    }

    readonly property string socialShareQml : "
        import QtQuick 2.3

        import wesual.Editor 1.0

        SocialShareButton {
            id : shareButton
            documents : [ product ]
            anchors.left : picker.right
            anchors.leftMargin : 10
            enabled : product && product.primaryImage &&
                      product.primaryImage.thumbnail !== \"\"
        }"

    signal navigateToProduct()

    implicitWidth  : parent ? parent.width : 100
    implicitHeight : items.height

    MediaLocationDialog {
        id : media

        selectMultiple : false
        onAccepted : {
            product.beginRecordChanges();
            product.primaryImage = result[0];
            product.endRecordChanges();
        }
    }

    Column {
        id : items

        readonly property int labelWidth   : Math.max(priceLabel.implicitWidth,
                                                    tagLabel.implicitWidth,
                                                    descritpionLabel.implicitWidth,
                                                    titleLabel.implicitWidth,
                                                    allergenLabel.implicitWidth) + 1
        readonly property int controlWidth : items.width - labelWidth - rowSpacing
        readonly property int rowSpacing   : 18
        width   : parent.width
        spacing : 9
        height  : childrenRect.height

        Item {
            id : pickerItem

            width :  parent.width
            height : picker.height

            ImagePickerButton {
                id : picker
                width : 110
                height : 110
                imageSource : product && product.primaryImage ?
                                             product.primaryImage.thumbnail : ""
                document : product ? product : null
                onSelectedImages : {
                    product.beginRecordChanges();
                    product.primaryImage = images[0];
                    product.endRecordChanges();
                }

                onNavigateToProduct : {
                    editor.navigateToProduct();
                }
            }

            UiToolButton {
                id : exchangeImage

                anchors {
                    left : picker.right
                    top  : p_.shareButton ? p_.shareButton.bottom : undefined
                    leftMargin : 10
                    topMargin  : 6
                }
                iconName  : "exchange-image"
                text      : qsTrId("26e3dacd662ab645")
                onClicked : media.show()
            }

            Component.onCompleted : {
                p_.shareButton = Qt.createQmlObject(socialShareQml, pickerItem);
            }
        }

        Repeater {
            model : [ ]
        }

        Row {
            spacing : items.rowSpacing

            UiLabel {
                id : priceLabel

                width : items.labelWidth
                text  : "Price"
                anchors {
                    verticalCenter : parent.verticalCenter
                    verticalCenterOffset : 1
                }
            }

            Row {
                id : currency

                spacing : 3

                Text {
                    anchors.verticalCenter : parent.verticalCenter
                    font {
                        family : UI.fontFamily(UI.PrimaryFont)
                        weight : UI.fontWeight(UI.SecondaryFont)
                        pixelSize : 14
                    }
                    text : {
                        if (!editor.product)
                            return "";

                        return UI.currencyString(
                                    product.context.country,
                                    UI.fontFamily(UI.PrimaryFont));
                    }
                }
                UiCondensedLineEdit {
                    id : productPrice

                    width : 80
                    bindable : true
                    text : product ? product.price.toLocaleString(
                                         Qt.locale(), "f", 2) : ""

                    onUpdateRequested : {
                        var price = newValue.replace(/,/g, ".");

                        if (!isNaN(price)) {
                            product.beginRecordChanges();
                            product.price = price;
                            product.endRecordChanges();
                        }
                    }
                }
            }
        }
        Row {
            spacing : items.rowSpacing

            UiLabel {
                id : tagLabel

                width : items.labelWidth
                text  : "Tags"
                anchors.baseline : tagPicker.baseline
            }
            UiDropDown {
                id : tagPicker

                width : items.controlWidth
                model : filteredTagListing.resultingDocuments
                hideCurrent : false
                currentIndex : -1

                delegate : Item {
                    implicitHeight : 25
                    implicitWidth  : tagPicker.width

                    TagIcon {
                        id : tagIcon

                        width  : 25
                        height : width
                        tag    : modelData ? modelData : null
                        tintColor : "black"
                        anchors {
                            left : parent.left
                            leftMargin : 5
                            verticalCenter : parent.verticalCenter
                        }
                    }
                    UiText {
                        text  : modelData ? modelData.name : ""
                        anchors {
                            left : tagIcon.right
                            leftMargin : 2
                            verticalCenter : parent.verticalCenter
                        }
                    }
                }
                labelDelegate : UiText {
                    text  : qsTrId("dedd1f8ecbeacd2a")
                    color : UiColors.getColor(UiColors.MediumLightGrey27)
                }
                onIndexPicked : {
                    product.addTag(model[index]);
                    currentIndex = -1;
                }
            }
        }
        Row {
            width   : parent.width
            spacing : items.rowSpacing
            visible : tagFlow.implicitWidth > 0

            UiLabel {
                width : items.labelWidth
                text  : ""
            }
            TagButtonFlow {
                id : tagFlow

                width            : items.controlWidth
                tagNamePixelSize : 14
                tagIconSize      : 25

                tagListing : {
                    if (!editor.product)
                        return [];

                    var allTags = editor.product.tags;
                    var foodserviceTags = [];

                    for (var i = 0; i < allTags.length; ++i) {
                        if (allTags[i].type === "tag:product:foodservice")
                            foodserviceTags.push(allTags[i]);
                    }

                    return foodserviceTags;
                }
                onRemoveTag : {
                    editor.product.beginRecordChanges();
                    editor.product.removeTag(tag);
                    editor.product.endRecordChanges();
                }
            }
        }

        Row {
            width   : parent.width
            spacing : items.rowSpacing

            UiLabel {
                id : allergenLabel

                width : items.labelWidth
                text  : qsTrId("3c6d3b9a1fbe063f")
                anchors.verticalCenter : parent.verticalCenter
            }
            FoodServiceAllergenButton {
                id : allergenButton

                currentProduct  : editor.product
                allergenListing : sortedAllergens.documents ?
                                  sortedAllergens.resultingDocuments : []
                maximumWidth    : items.controlWidth - 50
            }
        }
        Row {
            width   : parent.width
            spacing : items.rowSpacing

            UiLabel {
                id : titleLabel

                anchors.baseline : nameTextEdit.baseline
                width : items.labelWidth
                text  : qsTrId("48f5fac2f7ca2304")
            }
            UiCondensedLineEdit {
                id : nameTextEdit

                width           : Math.min(items.controlWidth, implicitWidth)
                placeHolderText : qsTrId("357e62e8cc4983da")
            }
            ControlLink {
                control          : nameTextEdit
                controlProperty  : "text"
                document         : editor.product
                documentProperty : "name"
            }
        }
        Row {
            width   : parent.width
            spacing : items.rowSpacing

            UiLabel {
                id : descritpionLabel

                anchors.baseline : descriptionTextEdit.baseline
                width : items.labelWidth
                text  :
                    qsTrId("632d35bb1ac3e18a")
            }
            UiCondensedTextEdit {
                id : descriptionTextEdit

                width           : Math.min(items.controlWidth, implicitWidth)
                placeHolderText : qsTrId("357e62e8cc4983da")
            }
            ControlLink {
                control          : descriptionTextEdit
                controlProperty  : "text"
                document         : editor.product
                documentProperty : "description"
            }
        }
    }

    TypeSorter {
        id : nameSorter

        property : "name"
    }

    TypeSorter {
        id : codeSorter

        property : "letterCode"
    }

    PersistentTypeListing {
        id : allergens

        context : product.context
        type    : "tag:allergen"
    }

    DocumentListSorter {
        id : sortedAllergens

        documents : allergens.documents
        sorters   : codeSorter
    }

    PersistentTypeListing {
        id : tagListing

        context : product.context
        type    : "tag:product:foodservice"
    }

    DocumentListSorter {
        id : sortedTags

        documents : tagListing.documents
        sorters   : nameSorter
    }
    DocumentListFilter {
        id : filteredTagListing

        documents : sortedTags.resultingDocuments
    }
}

